/*
 * Copyright European Commission's
 * Taxation and Customs Union Directorate-General (DG TAXUD).
 */
package eu.europa.ec.taxud.cesop.readers;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamConstants;
import javax.xml.stream.XMLStreamException;

import java.io.InputStream;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Stack;

import org.codehaus.stax2.XMLStreamReader2;
import org.codehaus.stax2.validation.XMLValidationException;
import org.codehaus.stax2.validation.XMLValidationSchema;

import com.ctc.wstx.stax.WstxInputFactory;

import static java.nio.charset.StandardCharsets.UTF_8;

/**
 * Wrapper for stream reader.
 */
public final class XmlStreamReaderWrapper {

    static final XMLInputFactory inputFactory;

    static {
        System.setProperty("javax.xml.accessExternalDTD", "");

        // Hardwire to woodstox. No SPI so the shaded library does not break anything in user apps.
        System.setProperty("org.codehaus.stax2.validation.XMLValidationSchemaFactory.w3c", "com.ctc.wstx.msv.W3CSchemaFactory");
        inputFactory = new WstxInputFactory();

        inputFactory.setProperty(XMLInputFactory.IS_SUPPORTING_EXTERNAL_ENTITIES, false);
        inputFactory.setProperty(XMLInputFactory.SUPPORT_DTD, false);
        inputFactory.setProperty(XMLInputFactory.IS_COALESCING, true);
    }

    private final XMLStreamReader2 streamReader;
    private final Stack<String> stack = new Stack<>();
    private int currentEventType = -1;
    private boolean markedAsPeek = false;

    public XmlStreamReaderWrapper(InputStream inputStream, XMLValidationSchema xsdSchema) throws XMLStreamException {
        this.streamReader = (XMLStreamReader2) inputFactory.createXMLStreamReader(inputStream, UTF_8.name());
        if (xsdSchema != null) {
            this.streamReader.validateAgainst(xsdSchema);
        }
    }

    /**
     * Mark as peek.
     */
    public void markAsPeek() {
        this.markedAsPeek = true;
    }

    /**
     * Returns true if there are more parsing events and false if there are no more events.
     * This method will return false if the current state of the XMLStreamReader is END_DOCUMENT.
     *
     * @return true if there are more events, false otherwise
     * @throws XMLStreamException in case of error while processing the XML content
     */
    public boolean hasNext() throws XMLStreamException {
        return this.markedAsPeek || this.streamReader.hasNext();
    }

    /**
     * Moves to the next element.
     * This method will throw an IllegalStateException if it is called after hasNext() returns false.
     *
     * @throws XMLStreamException in case of error while processing the XML content
     */
    public void moveToNextElement() throws XMLStreamException {
        if (this.markedAsPeek) {
            this.markedAsPeek = false;
        } else {
            try {
                this.currentEventType = this.streamReader.next();
            } catch (final XMLValidationException e) {
                // enhance default location, add path of elements pointing to the current position
                e.getValidationProblem().setLocation(new XmlLocation(e.getValidationProblem().getLocation(), stack));
                throw e;
            }
            if (isStartElement()) {
                stack.push(streamReader.getName().getLocalPart());
            } else if (isEndElement()) {
                stack.pop();
            }
        }
    }

    /**
     * Returns true if the current event type is a start element, false otherwise.
     *
     * @return true if the current event type is a start element.
     */
    public boolean isStartElement() {
        return this.currentEventType == XMLStreamConstants.START_ELEMENT;
    }

    /**
     * Returns true if the current event type is a characters element or a CDATA section, false otherwise.
     *
     * @return true if the current event type is a characters element or a CDATA section.
     */
    public boolean isCharactersElement() {
        return this.currentEventType == XMLStreamConstants.CHARACTERS
                || this.currentEventType == XMLStreamConstants.CDATA;
    }

    /**
     * Returns true if the current event type is an end element, false otherwise.
     *
     * @return true if the current event type is an end element.
     */
    public boolean isEndElement() {
        return this.currentEventType == XMLStreamConstants.END_ELEMENT;
    }

    /**
     * Goes to the next XML start element.
     *
     * @return true if a start element is found, false otherwise
     * @throws XMLStreamException in case of error while processing the XML content
     */
    public boolean goToNextStartElement() throws XMLStreamException {
        return this.goToNextStartElement(null);
    }

    /**
     * Goes to the next XML start element with specified QName.
     *
     * @param qName the name of the start element to search for
     * @return true if the start element is found, false otherwise
     * @throws XMLStreamException in case of error while processing the XML content
     */
    public boolean goToNextStartElement(final QName qName) throws XMLStreamException {
        while (this.streamReader.hasNext()) {
            this.moveToNextElement();
            if (this.isStartElement() && (qName == null || qName.equals(this.streamReader.getName()))) {
                return true;
            }
        }
        return false;
    }

    /**
     * Returns the current start tag name.
     *
     * @return the tag name
     * @throws XMLStreamException if the current event type is not a start element
     */
    public QName getStartElementName() throws XMLStreamException {
        if (this.isStartElement()) {
            return this.streamReader.getName();
        }
        throw new XMLStreamException("The current event is not a start element: " + this.currentEventType);
    }

    /**
     * Returns the current end tag name.
     *
     * @return the tag name
     * @throws XMLStreamException if the current event type is not an end element
     */
    public QName getEndElementName() throws XMLStreamException {
        if (this.isEndElement()) {
            return this.streamReader.getName();
        }
        throw new XMLStreamException("The current event is not an end element: " + this.currentEventType);
    }

    /**
     * Returns the element characters.
     *
     * @return the element characters
     * @throws XMLStreamException if the current event type is not a characters element
     */
    public String getCharacters() throws XMLStreamException {
        if (this.isCharactersElement()) {
            return this.streamReader.getText();
        }
        throw new XMLStreamException("The current event is not a characters element: " + this.currentEventType);
    }

    /**
     * Returns the list of attributes.
     *
     * @return a list of attributes (name, value)
     * @throws XMLStreamException if the current event type is not an attribute element
     */
    public Map<String, String> getAttributes() throws XMLStreamException {
        if (this.isStartElement()) {
            final Map<String, String> attributes = new LinkedHashMap<>();
            final int attributeCount = this.streamReader.getAttributeCount();
            for (int i = 0; i < attributeCount; i++) {
                final QName attributeName = this.streamReader.getAttributeName(i);
                final String attributeValue = this.streamReader.getAttributeValue(i);
                attributes.put(attributeName.getLocalPart(), attributeValue);
            }
            return attributes;
        }
        throw new XMLStreamException("The current event is not an attribute element: " + this.currentEventType);
    }

    void close() throws XMLStreamException {
        streamReader.close();
    }
}
